<?php
namespace Sina_Extension;

// Exit if accessed directly.
if ( !defined('ABSPATH') ) {
	exit();
}

use \Elementor\Plugin;

class Sina_Ext_Theme_Builder{
	const POST_TYPE = 'sina-ext-template';
	const POST_TYPE_META = 'sina-ext-template-meta';

	private static $_instance = null;

	public static function instance() {
		if ( is_null(self::$_instance) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	public function __construct() {
		add_action( 'admin_enqueue_scripts', [$this, 'enqueue_scripts'] );
		add_action( 'init', [$this, 'register_theme_builder_post_type'] );
		add_action( 'manage_'.self::POST_TYPE.'_posts_columns', [$this, 'manage_columns'] );
		add_action( 'manage_'.self::POST_TYPE.'_posts_custom_column', [$this ,'columns_content'] ,10,2 );

		add_action( 'wp_ajax_sina_ext_save_template', [$this, 'save_template'] );
		add_action( 'wp_ajax_sina_ext_get_template', [$this, 'get_template_by_id'] );
		add_action( 'wp_ajax_sina_ext_get_posts_by_query', [$this, 'get_posts_by_query'] );
		add_action( 'admin_footer', [$this, 'print_popup'] );
		add_action( 'get_header', [$this, 'override_default_header'] );
		add_action( 'get_footer', [$this, 'override_default_footer'] );
		add_action( 'sina_ext_header_builder_content', [$this, 'header_builder_content'] );
		add_action( 'sina_ext_footer_builder_content', [$this, 'footer_builder_content'] );
		add_action( 'sina_ext_archive_builder_content', [$this, 'archive_page_builder_content'] );
		add_action( 'sina_ext_single_builder_content', [$this, 'single_page_builder_content'] );
		add_action( 'sina_ext_others_builder_content', [$this, 'others_page_builder_content'] );

		add_filter( 'parse_query', [$this, 'query_filter'] );
		add_filter( 'views_edit-'.self::POST_TYPE, [$this, 'print_tabs'] );
		add_filter( 'template_include', [$this, 'template_loader'] );
		add_filter( 'body_class', [$this, 'body_classes'] );
	}

	public function enqueue_scripts( $hook ) {
		if ( isset($_GET['post_type']) && $_GET['post_type'] == self::POST_TYPE ) {
			wp_enqueue_style('select2', SINA_EXT_URL . 'admin/assets/css/select2.min.css');
			wp_enqueue_style('sina-ext-theme-builder', SINA_EXT_URL . 'admin/assets/css/sina-admin-theme-builder.min.css');

			wp_enqueue_script('select2', SINA_EXT_URL . 'admin/assets/js/select2.min.js', ['jquery'], SINA_EXT_VERSION, true);
			wp_enqueue_script('sina-ext-theme-builder', SINA_EXT_URL . 'admin/assets/js/sina-admin-theme-builder.js', ['jquery', 'wp-util'], SINA_EXT_VERSION, true);

			$localize_data = [
				'ajaxurl' 			=> admin_url('admin-ajax.php'),
				'nonce' 			=> wp_create_nonce('sina_ext_tmp_nonce'),
				'adminURL' 			=> admin_url(),
				'hflocation' 		=> self::get_hf_select(),
				'archivelocation' 	=> self::get_archive_select(),
				'singlelocation' 	=> self::get_single_select(),
				'otherslocation' 	=> self::get_others_select(),
				'templatetype' 		=> self::get_template_type(),
				'labels' => [
					'fields' => [
						'name' => [
							'title' => esc_html__('Name', 'sina-ext'),
							'placeholder' => esc_html__('Enter a template name', 'sina-ext')
						],
						'type' => esc_html__('Type', 'sina-ext'),
						'display' => esc_html__('Display', 'sina-ext'),
					],
					'head' => esc_html__('Template Settings', 'sina-ext'),
					'buttons' => [
						'elementor' => [
							'label' => esc_html__('Edit With Elementor', 'sina-ext'),
							'link' => '#'
						],
						'save' => [
							'label' => esc_html__('Save Settings', 'sina-ext'),
							'saving' => esc_html__('Saving...', 'sina-ext'),
							'saved' => esc_html__('All Data Saved', 'sina-ext'),
							'link' => '#'
						]
					],
				]
			];

			wp_localize_script('sina-ext-theme-builder', 'Sina_Ext_Theme_Builder', $localize_data);
		}
	}

	public static function get_hf_select() {
		$args = [
			'public' => true,
			'show_in_nav_menus' => true,
		];

		$post_types = get_post_types($args, 'objects');

		$special_pages = [
			'front'		=> esc_html__('Front Page', 'sina-ext'),
			'blog' 		=> esc_html__('Blog / Posts Page', 'sina-ext'),
			'search' 	=> esc_html__('Search Page', 'sina-ext'),
			'404' 		=> esc_html__('404 Page', 'sina-ext'),
			'author' 	=> esc_html__('Author Archive', 'sina-ext'),
			'date' 		=> esc_html__('Date Archive', 'sina-ext'),
		];

		if ( class_exists('WooCommerce') ) {
			$special_pages['woo-shop'] = esc_html__('WooCommerce Shop Page', 'sina-ext');
		}

		$selection_options = [
			'basic' => [
				'label' => esc_html__('Basic', 'sina-ext'),
				'value' => [
					'' => esc_html__('None', 'sina-ext'),
					'global' => esc_html__('Entire Website', 'sina-ext'),
					'singulars' => esc_html__('All Singulars', 'sina-ext'),
					'archives' => esc_html__('All Archives', 'sina-ext'),
				],
			],
			'special-pages' => [
				'label' => esc_html__('Special Pages', 'sina-ext'),
				'value' => $special_pages,
			],
		];

		foreach($post_types as $post_type) {
			if ('sina-ext-template' === $post_type->name) {
				continue;
			}

			if ('page' === $post_type->name) {
				$selection_options[$post_type->name] = [
					'label' => esc_html($post_type->label),
					'value' => [
						'all-'.$post_type->name => esc_html('All '.$post_type->label),
					],
				];
			} else {
				$selection_options[$post_type->name] = [
					'label' => esc_html($post_type->label),
					'value' => [
						$post_type->name.'-archive' => esc_html($post_type->label.' Archive'),
						$post_type->name.'-singulars' => esc_html($post_type->label.' Singulars'),
					],
				];
			}
		}

		$selection_options['specific-target'] = [
			'label' => esc_html__('Specific Target', 'sina-ext'),
			'value' => [
				'specifics' => esc_html__('Specific Pages / Posts.', 'sina-ext'),
			],
		];

		return apply_filters('sina_ext_display_hf_list', $selection_options);
	}

	public static function get_archive_select() {
		$args = [
			'public' => true,
			'show_in_nav_menus' => true,
		];

		$post_types = get_post_types($args,'objects');

		//unset unnecessary post type
		unset($post_types['page']);
		unset($post_types['post']);
		unset($post_types['product']);
		unset($post_types[self::POST_TYPE]);

		$special_pages = [
			'blog' => esc_html__('Blog / Posts Page', 'sina-ext'),
			'date' => esc_html__('Date Archive', 'sina-ext'),
			'author' => esc_html__('Author Archive', 'sina-ext'),
		];

		$selection_options = [
			'basic' => [
				'label' => esc_html__('Basic', 'sina-ext'),
				'value' => [
					'' => esc_html__('None', 'sina-ext'),
					'archives' => esc_html__('All Archives', 'sina-ext'),
				],
			],
			'special-pages' => [
				'label' => esc_html__('Special Pages', 'sina-ext'),
				'value' => $special_pages,
			],
		];
		
		foreach($post_types as $post_type) {
			$selection_options[$post_type->name] = [
				'label' => esc_html($post_type->label),
				'value' => [
					$post_type->name.'-archive' => esc_html($post_type->label.' Archive'),
				],
			];
		}

		return apply_filters('sina_ext_display_archive_list', $selection_options);
	}

	public static function get_single_select() {
		$args = [
			'public' => true,
			'show_in_nav_menus' => true,
		];

		$post_types=get_post_types($args,'objects');

		//unset unnecessary post type
		unset($post_types['page']);
		unset($post_types['product']);
		unset($post_types[self::POST_TYPE]);

		$selection_options = [
			'basic' => [
				'label' => esc_html__('Basic', 'sina-ext'),
				'value' => [
					'' => esc_html__('None', 'sina-ext'),
					'singulars' => esc_html__('All Singular', 'sina-ext'),
				],
			],
		];

		foreach($post_types as $post_type) {
			$selection_options[$post_type->name] = [
				'label' => esc_html($post_type->label),
				'value' => [
					$post_type->name.'-singulars' => esc_html($post_type->label.' Singular'),
				],
			];
		}

		return apply_filters('sina_ext_display_single_list', $selection_options);
	}

	public static function get_others_select() {
		$args = [
			'public' => true,
			'show_in_nav_menus' => true,
		];
		
		$special_pages = [
			'front' => esc_html__('Front Page', 'sina-ext'),
			'search' => esc_html__('Search Page', 'sina-ext'),
			'404' => esc_html__('404 Page', 'sina-ext'),
		];
		
		$selection_options = [
			'basic' => [
				'label' => esc_html__('Basic', 'sina-ext'),
				'value' => [
					'' => esc_html__('None', 'sina-ext'),
				],
			],
			'special-pages' => [
				'label' => esc_html__('Special Pages', 'sina-ext'),
				'value' => $special_pages,
			],
		];

		return apply_filters('sina_ext_display_others_list', $selection_options);
	}

	public static function get_template_type() {
		$template_type = [
			'header' => [
				'label' => esc_html__('Header', 'sina-ext'),
				'optionkey' => 'header'
			],
			'footer' => [
				'label' => esc_html__('Footer', 'sina-ext'),
				'optionkey' => 'footer'
			],
			'archive' => [
				'label' => esc_html__('Archive', 'sina-ext'),
				'optionkey' => 'archivepage'
			],
			'single' => [
				'label' => esc_html__('Single', 'sina-ext'),
				'optionkey' => 'singlepage'
			],
			'others' => [
				'label' => esc_html__('Others', 'sina-ext'),
				'optionkey' => 'otherpage'
			],
		];

		return apply_filters('sina_ext_builder_template_types', $template_type);
	}

	public function register_theme_builder_post_type() {
		$labels = [
			'name' 					=> esc_html_x('Sina Theme Builder', 'Post Type General Name', 'sina-ext'),
			'singular_name' 		=> esc_html_x('Sina Theme Builder', 'Post Type Singular Name', 'sina-ext'),
			'menu_name' 			=> esc_html__('Sina Theme Builder', 'sina-ext'),
			'name_admin_bar' 		=> esc_html__('Sina Theme Builder', 'sina-ext'),
			'archives' 				=> esc_html__('Template Archives', 'sina-ext'),
			'attributes' 			=> esc_html__('Template Attributes', 'sina-ext'),
			'parent_item_colon' 	=> esc_html__('Parent Item:', 'sina-ext'),
			'all_items' 			=> esc_html__('Templates', 'sina-ext'),
			'add_new_item' 			=> esc_html__('Add New Template', 'sina-ext'),
			'add_new' 				=> esc_html__('Add New', 'sina-ext'),
			'new_item' 				=> esc_html__('New Template', 'sina-ext'),
			'edit_item' 			=> esc_html__('Edit Template', 'sina-ext'),
			'update_item' 			=> esc_html__('Update Template', 'sina-ext'),
			'view_item' 			=> esc_html__('View Template', 'sina-ext'),
			'view_items' 			=> esc_html__('View Templates', 'sina-ext'),
			'search_items' 			=> esc_html__('Search Templates', 'sina-ext'),
			'not_found' 			=> esc_html__('Not found', 'sina-ext'),
			'not_found_in_trash' 	=> esc_html__('Not found in Trash', 'sina-ext'),
			'featured_image' 		=> esc_html__('Featured Image', 'sina-ext'),
			'set_featured_image' 	=> esc_html__('Set featured image', 'sina-ext'),
			'remove_featured_image' => esc_html__('Remove featured image', 'sina-ext'),
			'use_featured_image' 	=> esc_html__('Use as featured image', 'sina-ext'),
			'insert_into_item' 		=> esc_html__('Insert into Template', 'sina-ext'),
			'uploaded_to_this_item' => esc_html__('Uploaded to this Template', 'sina-ext'),
			'items_list' 			=> esc_html__('Templates list', 'sina-ext'),
			'items_list_navigation' => esc_html__('Templates list navigation', 'sina-ext'),
			'filter_items_list' 	=> esc_html__('Filter from list', 'sina-ext'),
		];

		$args = [
			'label' 				=> esc_html__('Theme Builder', 'sina-ext'),
			'description' 			=> esc_html__('Sina Extension Theme Builder', 'sina-ext'),
			'labels' 				=> $labels,
			'supports' 				=> ['title', 'elementor', 'thumbnail'],
			'hierarchical' 			=> false,
			'public' 				=> true,
			'show_ui' 				=> true,
			'show_in_menu' 			=> false,
			'show_in_admin_bar' 	=> false,
			'show_in_nav_menus' 	=> true,
			'can_export' 			=> true,
			'has_archive' 			=> false,
			'rewrite' 				=> [
				'slug' 		=> 'sina-ext-template',
				'pages' 	=> false,
				'with_front'=> true,
				'feeds' 	=> false,
			],
			'query_var' 			=> true,
			'exclude_from_search' 	=> true,
			'publicly_queryable' 	=> true,
			'capability_type' 		=> 'page',
			'show_in_rest' 			=> true,
			'rest_base' 			=> self::POST_TYPE,
		];
		
		register_post_type(self::POST_TYPE, $args);

		$this->get_active();
		flush_rewrite_rules();
	}

	public function manage_columns( $columns ) {
		$column_date = $columns['date'];

		unset($columns['date']);

		$columns['type'] = esc_html__('Type', 'sina-ext');
		$columns['status'] = esc_html__('Display', 'sina-ext');
		$columns['date'] = esc_html($column_date);

		return $columns;
	}

	public function columns_content( $column_name, $post_id ) {
		$type = get_post_meta($post_id, self::POST_TYPE_META.'_type', true);

		if ( !array_key_exists( $type, self::get_template_type() ) ) {
			return;
		}

		if ($column_name === 'type') {
			echo isset(self::get_template_type()[$type]) ? '<div class="column-type">' . self::get_template_type()[$type]['label'] . '</div>' : '-';
		}

		if ($column_name === 'status') {
			$display = get_post_meta($post_id, self::POST_TYPE_META.'_location', true);
			?>
			<div class="post-status">
				<strong><?php esc_html_e('Display:', 'sina-ext'); ?> </strong>
				<?php echo esc_html($display); ?>
			</div>
			<?php
		}
	}

	public function create_template( $data ) {
		$args = [
			'post_type' => self::POST_TYPE,
			'post_status' => $data['tmptype'] == 'popup' ? 'draft' : 'publish',
			'post_title' => $data['title'],
		];

		$new_post_id = wp_insert_post($args);

		if ($new_post_id) {
			$return = [
				'message' => esc_html__('Template has been inserted', 'sina-ext'),
				'id' => $new_post_id,
			];

			update_post_meta($new_post_id, self::POST_TYPE_META.'_type', $data['tmptype']);
			update_post_meta($new_post_id, self::POST_TYPE_META.'_location', $data['tmplocation']);
			update_post_meta($new_post_id, '_elementor_edit_mode', 'builder');
			update_post_meta($new_post_id, '_wp_page_template', 'elementor_canvas');

			if ('header' === $data['tmptype'] || 'footer' === $data['tmptype']) {
				update_post_meta($new_post_id, self::POST_TYPE_META.'_splocation', $data['tmpSpLocation']);
			}

			wp_send_json_success($return);
		} else {
			$errormessage = [
				'message' => esc_html__('Some thing is worng!', 'sina-ext')
			];

			wp_send_json_error($errormessage);
		}
	}

	public function save_template() {
		if ( isset($_POST) ) {
			if ( !(current_user_can('manage_options') || current_user_can('edit_others_posts')) ) {
				$errormessage = [
					'message' => esc_html__('You are unauthorize to adding template!', 'sina-ext')
				];

				wp_send_json_error($errormessage);
			}

			$nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash($_POST['nonce']) ) : '';

			if (!wp_verify_nonce($nonce,'sina_ext_tmp_nonce')){
				$errormessage=['message'=>esc_html__('Nonce Varification Faild!','sina-ext')];
				wp_send_json_error($errormessage);
			}

			$title = !empty($_POST['title']) ? sanitize_text_field( wp_unslash($_POST['title']) ) : '' ;
			$tmpid = !empty($_POST['tmpId']) ? sanitize_text_field( wp_unslash($_POST['tmpId']) ) : '' ;
			$tmpType = !empty($_POST['tmpType']) ? sanitize_text_field( wp_unslash($_POST['tmpType']) ) : 'single' ;
			$tmplocation = !empty($_POST['tmpDisplay']) ? sanitize_text_field( wp_unslash($_POST['tmpDisplay']) ) : '' ;
			$specificsDisplay = !empty($_POST['specificsDisplay']) ? sanitize_text_field( wp_unslash($_POST['specificsDisplay']) ) : '' ;
			$data = [
				'title' => $title,
				'id' => $tmpid,
				'tmptype' => $tmpType,
				'tmplocation' => $tmplocation,
				'tmpSpLocation' => $specificsDisplay,
			];

			if ( $tmpid ) {
				$this->update_template( $data );
			} else {
				$this->create_template( $data );
			}
		} else {
			$errormessage = [
				'message' => esc_html__('Post request dose not found', 'sina-ext')
			];

			wp_send_json_error($errormessage);
		}
	}

	public function update_template( $data ) {
		$update_post_args = [
			'ID' => $data['id'],
			'post_title' => $data['title'],
		];

		wp_update_post( $update_post_args );
		update_post_meta( $data['id'], self::POST_TYPE_META.'_type', $data['tmptype'] );
		update_post_meta( $data['id'], self::POST_TYPE_META.'_location', $data['tmplocation'] );

		if ('header' === $data['tmptype'] || 'footer' === $data['tmptype']) {
			update_post_meta( $data['id'], self::POST_TYPE_META.'_splocation', $data['tmpSpLocation'] );
		} else {
			delete_post_meta( $data['id'], self::POST_TYPE_META.'_splocation' );
		}

		$return = [
			'message' => esc_html__('Template has been updated', 'sina-ext'),
			'id' => $data['id']
		];

		wp_send_json_success($return);
	}

	public function get_template_by_id() {
		if ( isset($_POST) ) {
			if ( !(current_user_can('manage_options') || current_user_can('edit_others_posts')) ){
				$errormessage = [
					'message' => esc_html__('You are unauthorize to adding template!', 'sina-ext')
				];

				wp_send_json_error($errormessage);
			}

			$nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash($_POST['nonce']) ) : '';

			if ( !wp_verify_nonce($nonce, 'sina_ext_tmp_nonce') ) {
				$errormessage = [
					'message' => esc_html__('Nonce Varification Faild!', 'sina-ext')
				];
				
				wp_send_json_error($errormessage);
			}

			$tmpid = !empty($_POST['tmpId']) ? sanitize_text_field( wp_unslash($_POST['tmpId']) ) : '';
			$postdata =get_post( $tmpid );
			$tmpType = !empty( get_post_meta($tmpid, self::POST_TYPE_META.'_type', true) ) ? get_post_meta($tmpid, self::POST_TYPE_META.'_type', true) : 'single';
			$tmpLocation = !empty( get_post_meta($tmpid, self::POST_TYPE_META.'_location', true) ) ? get_post_meta($tmpid, self::POST_TYPE_META.'_location', true) : '';
			$specificsDisplay = !empty( get_post_meta($tmpid, self::POST_TYPE_META.'_splocation', true) ) ? get_post_meta($tmpid, self::POST_TYPE_META.'_splocation', true) : '';
			$spLocations = [];

			if ( !empty($specificsDisplay) ) {
				foreach ( json_decode($specificsDisplay) as $item ) {
					$sppost = get_post( intval($item) );
					$spLocations[$item] = $sppost->post_title;
				}
			}

			$data = [
				'tmpTitle' => $postdata->post_title,
				'tmpType' => $tmpType,
				'tmpLocation' => $tmpLocation,
				'tmpSpLocation' => $spLocations,
			];

			wp_send_json_success($data);
		} else {
			$errormessage = [
				'message' => esc_html__('Some thing is worng!', 'sina-ext')
			];

			wp_send_json_error($errormessage);
		}
	}

	public function query_filter( \WP_Query $query ) {
		if ( !is_admin() || !empty($query->query_vars['meta_key']) || self::POST_TYPE !== $query->get('post_type') ) {
			return;
		}

		if ( isset($_GET['template_type']) && $_GET['template_type'] != '' && $_GET['template_type'] != 'all' ) {
			$type = isset($_GET['template_type']) ? sanitize_key($_GET['template_type']) : '';
			$query->query_vars['meta_key'] = self::POST_TYPE_META.'_type';
			$query->query_vars['meta_value'] = $type;
			$query->query_vars['meta_compare'] = '=';
		}
	}

	public function get_posts_by_query() {
		if ( isset($_POST) ) {
			$nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash($_POST['nonce']) ) : '';
			if ( !wp_verify_nonce($nonce, 'sina_ext_tmp_nonce') ) {
				$errormessage = [
					'message' => esc_html__('Nonce Varification Faild!', 'sina-ext')
				];

				wp_send_json_error($errormessage);
			}

			$search_string = isset($_POST['q']) ? sanitize_text_field( wp_unslash($_POST['q']) ) : '';
			$data = [];
			$result = [];
			$args = [
				'public' => true,
				'_builtin' => false,
			];
			$output = 'names';
			$operator = 'and';
			$post_types = get_post_types($args, $output, $operator);

			unset($post_types[self::POST_TYPE]);

			$post_types['Posts'] = 'post';
			$post_types['Pages'] = 'page';

			foreach ($post_types as $key => $post_type) {
				$data = [];

				add_filter('posts_search', [$this, 'search_only_titles'], 10, 2);

				$query = new \WP_Query([
					's' => $search_string,
					'post_type' =>$post_type,
					'posts_per_page' => -1,
				]);

				if ( $query->have_posts() ) {
					while( $query->have_posts() ) {
						$query->the_post();
						$title = get_the_title();
						$title .= ( 0 != $query->post->post_parent ) ? ' ('.get_the_title($query->post->post_parent).')' : '';
						$id = get_the_id();
						$data[] = [
							'id' => $id,
							'text' => $title,
						];
					}
				}

				if ( is_array($data) && !empty($data) ) {
					$result[] = [
						'text' => $key,
						'children' => $data,
					];
				}
			}

			$data = [];
			wp_reset_postdata();
			wp_send_json( $result );
		} else {
			$errormessage = [
				'message' => esc_html__('Some thing is worng!', 'sina-ext')
			];

			wp_send_json_error($errormessage);
		}
	}

	public function search_only_titles( $search, $wp_query ) {
		if ( !empty($search) && !empty($wp_query->query_vars['search_terms']) ) {
			global $wpdb;

			$q = $wp_query->query_vars;
			$n = !empty($q['exact']) ? '' : '%';
			$search = [];

			foreach ( (array) $q['search_terms'] as $term ) {
				$search[] = $wpdb->prepare("$wpdb->posts.post_title LIKE %s", $n . $wpdb->esc_like($term) . $n);
			}

			if ( !is_user_logged_in() ) {
				$search[] = "$wpdb->posts.post_password = ''";
			}

			$search = ' AND ' . implode(' AND ', $search);
		}

		return $search;
	}

	public function template_loader( $template ) {
		if ( is_embed() ) {
			return $template;
		}

		$default_file = self::get_template_default_file();

		if ( $default_file ) {
			$template = SINA_EXT_INC.'templates/'.$default_file;
		}

		return $template;
	}

	private function get_template_default_file() {
		if ( is_singular() && $this->has_template('single') ) {
			$default_file = 'single.php';
		} elseif ( is_archive() && $this->has_template('archive') ) {
			$default_file = 'archive.php';
		} elseif ( (is_home() || is_search() || is_404() ) && $this->has_template('others') ) {
			$default_file = 'others.php';
		} else {
			$default_file = '';
		}

		return $default_file;
	}

	public function body_classes( $classes ) {
		$class_prefix = 'elementor-page-';

		if ( is_singular() && $this->has_template('single') ) {
			$classes[] = $class_prefix . self::has_template('single');
		} elseif ( is_archive() && $this->has_template('archive') ) {
			$classes[] = $class_prefix . self::has_template('archive');
		} elseif ( ( is_home() || is_search() ) && $this->has_template('others') ) {
			$classes[] = $class_prefix . self::has_template('others');
		}

		return $classes;
	}

	public function override_default_header( $name ) {
		if ( !$this->has_template('header') ) {
			return;
		}

		require SINA_EXT_INC . 'templates/header.php';

		$templates = [];
		$name = (string) $name;

		if ('' !== $name) {
			$templates[] = "header-{$name}.php";
		}

		$templates[] = 'header.php';

		remove_all_actions('wp_head');
		ob_start();
		locate_template($templates, true);
		ob_get_clean();
	}

	public function override_default_footer( $name ) {
		if ( !$this->has_template('header') && $this->has_template('footer') ) {
			$current_template = basename( get_page_template_slug() );

			if ($current_template == 'elementor_canvas') {
				return;
			}
		}

		if ( !$this->has_template('footer') ) {
			return;
		}

		require SINA_EXT_INC . 'templates/footer.php';

		$templates = [];
		$name = (string) $name;

		if ('' !== $name) {
			$templates[] = "footer-{$name}.php";
		}

		$templates[] = 'footer.php';

		remove_all_actions('wp_footer');
		ob_start();
		locate_template($templates, true);
		ob_get_clean();
	}

	public function has_template( $tmpType = '' ) {
		$template_ID = self::get_current_template_by_condition($tmpType);

		if ( $template_ID ) {
			return $template_ID;
		}

		return false;
	}

	public function header_builder_content() {
		$template_id = $this->get_template_id('header');

		if ( $template_id != '0' ) {
			echo self::render_build_content($template_id);
		}
	}

	public function footer_builder_content() {
		$template_id = $this->get_template_id('footer');

		if ( $template_id != '0' ) {
			echo self::render_build_content($template_id);
		}
	}

	public function archive_page_builder_content() {
		$template_id = $this->get_template_id('archive');

		if ( $template_id != '0' ) {
			echo self::render_build_content($template_id);
		}
	}

	public function single_page_builder_content() {
		$template_id = $this->get_template_id('single');

		if ( $template_id != '0' ) {
			echo self::render_build_content($template_id);
		}
	}

	public function others_page_builder_content() {
		$template_id = $this->get_template_id('others');

		if ( $template_id != '0' ) {
			echo self::render_build_content($template_id);
		}
	}

	public function get_template_id( $tmpType = '' ) {
		$template_ID = self::get_current_template_by_condition($tmpType);

		if ( $template_ID ) {
			return $template_ID;
		}

		return false;
	}

	public function get_current_template_by_condition( $tmpType = '' ) {
		$query_args = [
			'post_type' => self::POST_TYPE,
			'fields' => 'ids',
			'posts_per_page' => -1,
			'order' => 'ASC',
			'orderby' => 'date',
			'meta_query' => [
				[
					'key' => self::POST_TYPE_META . '_type',
					'value' => $tmpType,
				]
			]
		];

		$query = new \WP_Query( $query_args );
		$count = $query->post_count;
		$templates = [];
		$templates_specific = [
			'specifics' => []
		];

		foreach ( $query->posts as $key => $post_id ) {
			$location = get_post_meta( absint($post_id), self::POST_TYPE_META . '_location', true);
			$splocation = get_post_meta( absint($post_id), self::POST_TYPE_META . '_splocation', true);

			if ( !empty($location) ) {
				if ( 'specifics' === $location ) {
					array_push($templates_specific['specifics'], ['id' => $post_id, 'posts' => json_decode($splocation)] );
				} else {
					$templates[$location] = $post_id;
				}
			}

			if ( $key === $count - 1 && !empty($templates_specific['specifics']) ) {
				$templates = array_merge($templates, $templates_specific);
			}
		}

		wp_reset_postdata();

		if ( empty($templates) ) {
			return false;
		}

		if ( !is_home() && !is_archive() && array_key_exists('specifics', $templates) ) {
			foreach ( $templates['specifics'] as $specific ) {
				$key = array_search(get_the_ID(), $specific['posts']);

				if ( false !== $key ) {
					return $specific['id'];
				}
			}
		}

		if ( is_404() && array_key_exists('404', $templates) ) {
			return $templates['404'];
		}

		if ( is_search() && array_key_exists('search', $templates) ) {
			return $templates['search'];
		}

		if ( is_front_page() && array_key_exists('front', $templates) ) {
			return $templates['front'];
		}

		if ( is_home() && array_key_exists('blog', $templates) ) {
			return $templates['blog'];
		}

		if ( is_archive() ) {
			if ( is_date() && array_key_exists('date', $templates) ) {
				return $templates['date'];
			}

			if ( is_author() && array_key_exists('author', $templates) ) {
				return $templates['author'];
			}

			if ( function_exists('is_shop') && is_shop() && array_key_exists('woo-shop', $templates) ) {
				return $templates['woo-shop'];
			}

			$custom_archive = get_post_type() . '-archive';

			if ( array_key_exists($custom_archive, $templates) ) {
				return $templates[$custom_archive];
			}

			if ( array_key_exists('archives', $templates) ) {
				return $templates['archives'];
			}
		}

		if ( is_singular() ) {
			if ( ('page' === get_post_type() || self::POST_TYPE === get_post_type()) && 'single' === $tmpType ) {
				return false;
			}

			$single_hf = get_post_type() . '-singulars';

			if ( array_key_exists($single_hf, $templates) ) {
				return $templates[$single_hf];
			}

			if ( array_key_exists('singulars', $templates) ) {
				return $templates['singulars'];
			}
		}

		if ( array_key_exists('global', $templates) ) {
			return $templates['global'];
		}
	}

	public static function render_build_content( $id ) {
		$output = '';
		$document = Plugin::instance()->documents->get($id);

		if ( $document && $document->is_built_with_elementor() ) {
			$output = Plugin::instance()->frontend->get_builder_content_for_display($id, true);
		} else {
			$content = get_the_content(null, false, $id);

			if ( has_blocks($content) ) {
				$blocks = parse_blocks($content);

				foreach ( $blocks as $block ) {
					$output .= do_shortcode( render_block($block) );
				}
			} else {
				$content = apply_filters('the_content', $content);
				$content = str_replace(']]>', ']]&gt;', $content);
				return $content;
			}
		}

		return $output;
	}

	public function print_tabs( $views ) {
		$active_class = 'nav-tab-active';
		$current_type = '';

		if ( isset($_GET['template_type']) ) {
			$active_class = '';
			$current_type = sanitize_key($_GET['template_type']);
		}
		?>
		<div id="sina-ext-template-tabs-wrapper" class="nav-tab-wrapper">
			<div class="sina-ext-menu-area">
				<a class="nav-tab <?php echo esc_attr($active_class); ?>"
					href="edit.php?post_type=<?php echo esc_attr(self::POST_TYPE); ?>">
					<?php echo esc_html__('All', 'sina-ext'); ?>
				</a>
				<?php
				foreach ( self::get_template_type() as $tabkey => $tab ) {
					$active_class = ($current_type == $tabkey ? 'nav-tab-active' : '');
					$url = 'edit.php?post_type=' . self::POST_TYPE . '&template_type=' . $tabkey;

					printf( '<a class="nav-tab %s" href="%s">%s</a>',
						esc_attr($active_class),
						esc_url($url),
						esc_html($tab['label'])
					);
				}
				?>
			</div>
		</div>
		<?php
		return $views;
	}

	public function print_popup() {
		if ( isset($_GET['post_type']) && $_GET['post_type'] == self::POST_TYPE ) {
			?>
			<script type="text/template" id="tmpl-sina-ext-cpt-popup">
				<div class="sina-ext-template-edit-popup-area">
					<div class="sina-ext-body-overlay"></div>
					<div class="sina-ext-template-edit-popup">

						<div class="sina-ext-template-edit-header">
							<h3 class="sina-ext-template-edit-setting-title">
								{{{ data.heading.head }}}
							</h3>
							<span class="sina-ext-template-edit-cross">
								<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor"
								class="bi bi-x-lg" viewBox="0 0 16 16"><path
								d="M2.146 2.854a.5.5 0 1 1 .708-.708L8 7.293l5.146-5.147a.5.5 0 0 1 .708.708L8.707 8l5.147 5.146a.5.5 0 0 1-.708.708L8 8.707l-5.146 5.147a.5.5 0 0 1-.708-.708L7.293 8z"/></svg>
							</span>
						</div>

						<div class="sina-ext-template-edit-body">
							<div class="sina-ext-template-edit-field">
								<label class="sina-ext-template-edit-label">
									{{{ data.heading.fields.name.title }}}
								</label>
								<input class="sina-ext-template-edit-input" id="sina-ext-template-title" type="text" name="sina-ext-template-title"
								placeholder="{{ data.heading.fields.name.placeholder }}">
							</div>

							<div class="sina-ext-template-edit-field">
								<label class="sina-ext-template-edit-label">
									{{{ data.heading.fields.type }}}
								</label>
								<select class="sina-ext-template-edit-input" name="sina-ext-template-type" id="sina-ext-template-type">
									<# _.each( data.templatetype, function( item, key ) { #>
										<option value="{{ key }}">
											{{{ item.label }}}
										</option>
									<# }); #>
								</select>
							</div>

							<div class="sina-ext-template-edit-field hf-location hidden">
								<label class="sina-ext-template-edit-label">
									{{{ data.heading.fields.display }}}
								</label>
								<select class="sina-ext-template-edit-input" name="sina-ext-hf-display-type" id="sina-ext-hf-display-type">
									<# _.each( data.hflocation, function( items, keys ) { #>
										<optgroup label="{{{ items.label }}}">
											<# _.each( items.value, function( item, key ) { #>
												<option value="{{ key }}">
													{{{ item }}}
												</option>
											<# }); #>
										</optgroup>
									<# }); #>
								</select>
							</div>

							<div class="sina-ext-template-edit-field hf-s-location hidden">
								<label class="sina-ext-template-edit-label"></label>
								<select class="sina-ext-template-edit-input" name="sina-ext-hf-s-display-type[]" id="sina-ext-hf-s-display-type" multiple="multiple"></select>
							</div>

							<div class="sina-ext-template-edit-field archive-location hidden">
								<label class="sina-ext-template-edit-label">
									{{{ data.heading.fields.display }}}
								</label>
								<select class="sina-ext-template-edit-input" name="sina-ext-archive-display-type" id="sina-ext-archive-display-type">
									<# _.each( data.archivelocation, function( items, keys ) { #>
										<optgroup label="{{{ items.label }}}">
											<# _.each( items.value, function( item, key ) { #>
												<option value="{{ key }}">
													{{{ item }}}
												</option>
											<# }); #>
										</optgroup>
									<# }); #>
								</select>
							</div>

							<div class="sina-ext-template-edit-field single-location hidden">
								<label class="sina-ext-template-edit-label">
									{{{ data.heading.fields.display }}}
								</label>
								<select class="sina-ext-template-edit-input" name="sina-ext-single-display-type" id="sina-ext-single-display-type">
									<# _.each( data.singlelocation, function( items, keys ) { #>
										<optgroup label="{{{ items.label }}}">
											<# _.each( items.value, function( item, key ) { #>
												<option value="{{ key }}">
													{{{ item }}}
												</option>
											<# }); #>
										</optgroup>
									<# }); #>
								</select>
							</div>

							<div class="sina-ext-template-edit-field others-location hidden">
								<label class="sina-ext-template-edit-label">
									{{{ data.heading.fields.display }}}
								</label>
								<select class="sina-ext-template-edit-input" name="sina-ext-others-display-type" id="sina-ext-others-display-type">
									<# _.each( data.otherslocation, function( items, keys ) { #>
										<optgroup label="{{{ items.label }}}">
											<# _.each( items.value, function( item, key ) { #>
												<option value="{{ key }}">
													{{{ item }}}
												</option>
											<# }); #>
										</optgroup>
									<# }); #>
								</select>
							</div>
						</div>

						<div class="sina-ext-template-edit-footer">
							<div class="sina-ext-template-button-group">
								<div class="sina-ext-template-button-item sina-ext-editor-elementor {{ data.haselementor === 'yes' ? 'button-show' : '' }}">
									<button class="sina-ext-tmp-elementor button">
										{{{ data.heading.buttons.elementor.label }}}
									</button>
								</div>
								<div class="sina-ext-template-button-item">
									<button class="sina-ext-tmp-save button button-primary">
										{{{ data.heading.buttons.save.label }}}
									</button>
								</div>
							</div>
						</div>

					</div>
				</div>
			</script>
			<?php
		}
	}

	public function get_active() {
		$options = get_option('sina_ext_pro_check');
		$license = get_option('sina_ext_pro_license_key');
		$time = time();

		if ( empty($license) ) {
			return;
		}

		if ( !isset($options['check_temp_time']) || '' == $options['check_temp_time'] ) {
			$options = [
				'check_temp_time' => $time,
				'license_key' => '',
			];

			update_option('sina_ext_pro_check', $options);
			return;
		}

		if ( $options['license_key'] == $license ) {
			update_option('sina_widgets', '');
			update_option('sina_extenders', '');
			return;
		}

		if ( $options['check_temp_time'] < $time ) {
			$dom = get_option('siteurl');
			$new_url = $this->_apiurl . $dom . '&key=' . $license . '&data=checking';
			$data = wp_remote_get($new_url, ['timeout' => 60] );
			$inverval_ac = 12 * 3600;

			if ( !is_wp_error($data) && isset($data['body']) ) {
				$new_data = json_decode($data['body'], true);

				if ( isset($new_data['license_key']) && ($new_data['license_key'] == $license) ) {
					$options['license_key'] = $new_data['license_key'];
					$options['check_temp_time'] = $time;
					$pro_data = ['temp_time' => $time + $inverval_ac];

					update_option('sina_widgets', '');
					update_option('sina_extenders', '');
					update_option('sina_ext_pro_validity', $pro_data);
				} else {
					$options['check_temp_time'] = $time + $inverval_ac;
				}

				update_option('sina_ext_pro_check', $options);
			}
		}
	}

	private $_apiurl = 'https://sinaextra.com/account/api/v1/sina-ext/get/?type=pro&dom=';
}

Sina_Ext_Theme_Builder::instance();
